package de.christofreichardt.scala.krypto.algorithms

import de.christofreichardt.diagnosis.AbstractTracer
import de.christofreichardt.diagnosis.TracerFactory
import de.christofreichardt.scala.krypto.Constants
import scala.annotation.tailrec

class SieveOfEratosthenes(val limit: Int) extends Algorithm[Int, Seq[Int]](limit) {
	val rootOfLimit = scala.math.sqrt(limit).toInt + 1
	
  @tailrec
	private def sieve(prime: Int, multiples: Set[Int]): Set[Int] = {
    val tracer = getCurrentTracer
    tracer.out().printfIndentln("prime = %d", prime: Integer)
    tracer.out().flush()
    
    val nextMultiples = multiples ++ new Range(prime + prime, limit + prime, prime)
    
    def searchNextPrime(primeCandidate: Int): Int = {
      if (!nextMultiples.contains(primeCandidate)) primeCandidate
      else searchNextPrime(primeCandidate + 1)
    }
    
    if (prime >= rootOfLimit) nextMultiples
    else sieve(searchNextPrime(prime + 1), nextMultiples)
	}
	
	protected def calculate: Seq[Int] = {
	  withTracer("Seq[Int]", this, "calculate") {
	    val multiples = sieve(2, Set.empty)
	    val primes = 
	      (for {primeCandidate <- 2 to limit} yield if (!multiples.contains(primeCandidate)) primeCandidate else -1).filter(n => n > 0)
	    primes
	  }
	}
	
	override def crossCheck(): Boolean = {
	  outcome.forall(prime => BigInt(prime).isProbablePrime(Constants.certainty))
	}
  
  override def getCurrentTracer(): AbstractTracer = {
    try {
      TracerFactory.getInstance().getTracer("TestTracer")
    }
    catch {
      case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
    }
  }
}