package de.christofreichardt.scala.krypto.algorithms

import de.christofreichardt.scala.krypto.PrimeBase
import de.christofreichardt.diagnosis.AbstractTracer
import de.christofreichardt.diagnosis.TracerFactory
import de.christofreichardt.scala.krypto.Constants
import scala.annotation.tailrec

class PrimeFactorization(val number: BigInt) extends Algorithm[BigInt,List[Tuple2[BigInt,Int]]](number) {
  val maxExponent: Int = 4096
  
  @tailrec
  private def factorize(n: BigInt, factors: List[Tuple2[BigInt,Int]], primes: List[Int]): List[Tuple2[BigInt,Int]] = {
    
    def checkPrime(p: Int): Int = {
//      withTracer("Int", this, "checkPrime(p: Int)") {
//			  val tracer = getCurrentTracer
//			  tracer.out().printfIndentln("p = %d", int2Integer(p))
			  val exponents = Range(0, maxExponent)
			  val exponent = exponents.view.map(exponent => BigInt(p).pow(exponent)).indexWhere(power => n % power != 0) - 1
			  exponent
//      }
    }
    
//    withTracer("List[Tuple2[BigInt,Int]]", this, "factorize(n: BigInt, factors: List[Tuple2[BigInt,Int]], primes: List[Int])") {
//		  val tracer = getCurrentTracer
//		  tracer.out().printfIndentln("n = %s", n)
//		  tracer.out().printfIndentln("factors = %s", factors)
//		  tracer.out().printfIndentln("primes = %s", primes)
		  val exponent = checkPrime(primes.head)
//		  tracer.out().printfIndentln("exponent = %d", int2Integer(exponent))
		  val primePower = BigInt(primes.head).pow(exponent)
		  val next = n/primePower
	    val nextFactors = 
	      if (exponent != 0) (BigInt(primes.head), exponent) :: factors
	      else factors
	    if (next == BigInt(1)) nextFactors
	    else if (primes.tail == Nil) (next, 1) :: nextFactors
	    else factorize(next, nextFactors, primes.tail)
//    }
  }
  
	protected def calculate: List[Tuple2[BigInt,Int]] = {
	  withTracer("List[Tuple2[BigInt,Int]]", this, "calculate()") {
	    factorize(number, List.empty, PrimeBase.primes.view.takeWhile(prime => BigInt(prime)*BigInt(prime) <= number).toList)
	  }
	}
	
  lazy val productCheck = outcome.foldLeft(BigInt(1))((r,c) => r*c._1.pow(c._2)) == number
  lazy val primeCheck = outcome.forall(primePower => primePower._1.isProbablePrime(Constants.certainty))
	
	override def crossCheck(): Boolean = {
	  withTracer("Boolean", this, "crossCheck()") {
	    productCheck && primeCheck
	  }
	}
  
  override def getCurrentTracer(): AbstractTracer = {
    try {
      TracerFactory.getInstance().getTracer("TestTracer")
    }
    catch {
      case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
    }
  }
}