package de.christofreichardt.scala.krypto.algorithms

import de.christofreichardt.diagnosis.AbstractTracer
import de.christofreichardt.diagnosis.TracerFactory
import de.christofreichardt.scala.krypto.PrimeBase
import de.christofreichardt.scala.krypto.LCMBase

/**
 * Applies Pollards (p-1)-method to factorize n. TODO: this doesn't work for small n
 */
class PMinusOneMethod(val n: BigInt) extends Algorithm[BigInt, Option[BigInt]](n) {
  
  val coPrimesCount = 100
  val coPrimes = PrimeBase.primes.filter(prime => n.gcd(BigInt(prime)) == 1).take(coPrimesCount)
  require(!coPrimes.isEmpty, "Didn't find any coprimes to " + n + ".")
  
  lazy val cofactor = if (outcome.isDefined) Some(n/outcome.get) else None
  
  protected def firstStage(C: BigInt): Option[BigInt] = {
	  withTracer("BigInt", this, "firstStage()") {
	    val tracer = getCurrentTracer()
//	    tracer.out().printfIndentln("C = %s", C)
	    tracer.out().printfIndentln("coPrimes = %s", coPrimes)
	    val b: Option[BigInt] = coPrimes.view.map(coPrime => {
	      tracer.out().printfIndentln("coPrime = %d", int2Integer(coPrime))
	      BigInt(coPrime).modPow(C, n) - BigInt(1)
	    }).find(b => b != 0)
//	    if (b.isEmpty)
//	      throw new ArithmeticException("All coPrimes give d == " + n + ".")
	    if (b.isDefined) {
		    tracer.out().printfIndentln("b = %s", b.get)
		    Some(b.get.gcd(n))
	    }
	    else
	      None
	  }
  }

  override protected def calculate: Option[BigInt] = {
    withTracer("Option[BigInt]", this, "calculate()") {
	    val tracer = getCurrentTracer()
	    tracer.out().printfIndentln("n = %s", n)

	      LCMBase.lcms.view.zipWithIndex.map(C => {
	        tracer.out().printfIndentln("C[%d] = %s", C._2: Integer, C._1)
	        firstStage(C._1)
	      }).find(d => d.isDefined && d.get != BigInt(1)).flatMap(d => if (d.isDefined) Some(d.get) else None)
    }
  }

	override def crossCheck(): Boolean = {
	  if (outcome.isEmpty) throw new ArithmeticException("(p-1)-method couldn't factor " + n)
	  else n.mod(outcome.get) == BigInt(0)
	}

	override def getCurrentTracer(): AbstractTracer = {
    try {
      TracerFactory.getInstance().getTracer("TestTracer")
    } 
    catch {
      case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
    }
  }
}