package de.christofreichardt.scala.krypto.algorithms

import de.christofreichardt.diagnosis.AbstractTracer
import de.christofreichardt.diagnosis.TracerFactory


/**
 * Computes the order x of a given group element a, that is a^x mod p = 1 and x is minimal with that property
 */
class OrderOfElement (triple: Tuple3[BigInt,BigInt,List[Tuple2[BigInt,Int]]]) 
extends Algorithm[Tuple3[BigInt,BigInt,List[Tuple2[BigInt,Int]]], BigInt](triple) {
  val element: BigInt = triple._1
  val prime: BigInt = triple._2
  val primeFactors: List[Tuple2[BigInt,Int]] = triple._3
  
  lazy val primePowersList =
    this.primeFactors.map(factor => {
      val tracer = getCurrentTracer()
      tracer.out.printfIndentln("factor = %s", factor)
      (for (exponent <- 0 to factor._2) yield factor._1.pow(exponent)).toList
    })
    
  lazy val divisors = generateDivisors(primePowersList).sortWith((n1,n2) => n1 < n2)
  
  def generateDivisors(primePowersList: List[List[BigInt]]): List[BigInt] = {
    withTracer("List[BigInt]", this, "generateDivisors()") {
      val tracer = getCurrentTracer()
      
      (primePowersList: @unchecked) match {
        case primePowers :: Nil => primePowers
        case primePowers :: remainingPrimePowersList => {
          val divisors = generateDivisors(remainingPrimePowersList)
          for {
            factor1 <- primePowers
            factor2 <- divisors
          } yield factor1*factor2
        }
      }
    }
  }

  def calculate: BigInt = {
    withTracer("BigInt", this, "calculate()") {
      val tracer = getCurrentTracer()
      tracer.out.printfIndentln("element = %s", element)
      tracer.out.printfIndentln("prime = %s", prime)
      tracer.out.printfIndentln("primeFactors = %s", primeFactors)
      tracer.out.printfIndentln("primePowersList = %s", primePowersList)
      tracer.out.printfIndentln("divisors = %s", divisors)
      tracer.out.printfIndentln("divisors = %d", int2Integer(divisors.size))
      
      val optionalOrder =
	      divisors.find(divisor => {
	        val power = element.modPow(divisor, prime)
	        tracer.out.printfIndentln("%s^%s = %s (mod %s)", element, divisor, power, prime)
	        power == BigInt(1)
	      })
	    assert(optionalOrder.isDefined, "Order calculation failed.")
      optionalOrder.get
    }
  }
	
	override def crossCheck(): Boolean = {
	  withTracer("Boolean", this, "crossCheck()") {
	    element.modPow(outcome, prime) == BigInt(1) // && (BigInt(1).until(outcome)).forall(check => element.modPow(check, prime) != BigInt(1))
	  }
	}
  
  override def getCurrentTracer(): AbstractTracer = {
    try {
      TracerFactory.getInstance().getTracer("TestTracer")
    }
    catch {
      case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
    }
  }
}