package de.christofreichardt.scala.krypto.algorithms

import scala.util.Random
import de.christofreichardt.diagnosis.AbstractTracer
import de.christofreichardt.diagnosis.TracerFactory

/**
 * Computes a generator g for the multiplicative group Z(p) according to Shoups 
 * 'A Computational Introduction to Number Theory and Algebra', §11.1
 * 
 */
class GeneratorSearch(val prime: BigInt) extends Algorithm[BigInt,BigInt](prime) {
  val order = prime - 1
  
  lazy val primeFactorsOfOrder = {
    val tracer = getCurrentTracer
    val primeFactorization = new PrimeFactorization(order)
    assert(primeFactorization.crossCheck, "Primefactorization failed.")
    tracer.out().printfIndentln("primeFactors = %s", primeFactorization.outcome)
    primeFactorization.outcome
  }
  
  def generateRandomBigInts: Stream[BigInt] = {
    val tracer = getCurrentTracer
    val l = prime.bitLength
    val random = new Random
    def randomBigInt: BigInt = {
      val candidate = BigInt(l, random)
      if (candidate < prime  && candidate != BigInt(0)) candidate
      else randomBigInt
    }
    val next = randomBigInt
    tracer.out().printfIndentln("next = %s", next)
    Stream.cons(next, generateRandomBigInts)
  }
  
	def calculate: BigInt = {
	  withTracer("BigInt", this, "calculate()") {
	    val tracer = getCurrentTracer
	    val gammas =
	      for {
	        primeFactors <- primeFactorsOfOrder
	        q = primeFactors._1
	        e = primeFactors._2
	        unit0 = tracer.out().printfIndentln("q = %s", q)
	        alpha = generateRandomBigInts.find(alpha => alpha.modPow(order/q, prime) != BigInt(1))
	        unit1 = tracer.out().printfIndentln("alpha = %s", alpha)
	      } yield {
	        val gamma = alpha.get.modPow(order/q.pow(e), prime)
	        assert(gamma.modPow(q.pow(e - 1), prime) != BigInt(1))
	        gamma
	      }
      tracer.out().printfIndentln("gammas = %s", gammas)
      gammas.foldLeft(BigInt(1))((r,c) => (r*c).mod(prime))
    }
	}
	
	override def crossCheck(): Boolean = {
	  val orderOfElement = new OrderOfElement(outcome, prime, primeFactorsOfOrder)
	  orderOfElement.outcome == prime - 1  &&  orderOfElement.crossCheck
	}
  
  override def getCurrentTracer(): AbstractTracer = {
    try {
      TracerFactory.getInstance().getTracer("TestTracer")
    }
    catch {
      case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
    }
  }
}