package de.christofreichardt.scala.krypto.algorithms

package exec {

  import java.io.File

  import scala.xml.Atom
  import scala.xml.Elem
  import scala.xml.Node
  import scala.xml.NodeSeq
  import scala.xml.Text
  import scala.xml.TopScope

  import de.christofreichardt.diagnosis.AbstractTracer
  import de.christofreichardt.diagnosis.LogLevel
  import de.christofreichardt.diagnosis.TracerFactory
  import de.christofreichardt.scala.diagnosis.Tracing
  import de.christofreichardt.scala.krypto.DLProblem

  trait AlgorithmExec[U, V, W] extends Tracing {
    type SuitableAlgorithm <: Algorithm[U, V]

    lazy val suitableAlgorithm = createAlgorithmInstance(input())
    lazy val theResult = compute(suitableAlgorithm)
    lazy val correct = suitableAlgorithm.crossCheck

    def input(): U
    def output(): W
    def printDescription(): W
    def compute(algorithm: SuitableAlgorithm): V = algorithm.outcome
    def createAlgorithmInstance(parameter: U): SuitableAlgorithm
    def error(): W

    def process(): W = {
      withTracer("Unit", this, "process()") {
        printDescription()
        if (correct)
          output()
        else
          error()
      }
    }

    override def getCurrentTracer(): AbstractTracer = {
      try {
        TracerFactory.getInstance().getCurrentPoolTracer()
      } 
      catch {
        case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
      }
    }
  }

  class PohligHellmanTask(taskNode: Node, className: String, extensions: Boolean) extends AlgorithmExec[DLProblem, Option[BigInt], Elem] {
    type SuitableAlgorithm = PohligHellmanAlgo[_]

    override def input(): DLProblem = {
      withTracer("Elem", this, "input()") {
        val tracer = getCurrentTracer()
        val argsNode = (taskNode \ "args").head
        new DLProblem(BigInt((argsNode \ "a").text), BigInt((argsNode \ "b").text), BigInt((argsNode \ "p").text), BigInt((argsNode \ "order").text))
      }
    }

    override def output(): Elem = {
      withTracer("Unit", this, "output(result: Option[BigInt])") {
        val tracer = getCurrentTracer()

        val solution = theResult match {
          case Some(x) => (x.toString(), true)
          case None => ("No Solution.", false)
        }

        val resultElement =
          <job ref={ taskNode.attribute("id").get }>
            {
              if (extensions && solution._2) {
                <x>{ solution._1 }</x>
                <extensions>
                  <primefactors>
                    {
                      val test = suitableAlgorithm.primeFactors.zip(suitableAlgorithm.listOfcoefficients)
                      for (primefactor <- test) yield <primefactor>
                                                        <prime>{ primefactor._1._1 }</prime>
                                                        <exponent>{ primefactor._1._2 }</exponent>
                                                        <coefficients>{ primefactor._2.reverse.mkString(",") }</coefficients>
                                                      </primefactor>
                    }
                  </primefactors>
                  <equations>
                    {
                      for (equation <- suitableAlgorithm.equations.unzip._2) yield <equation>{ "x = " + equation._1 + " (mod " + equation._2 + ")" }</equation>
                    }
                  </equations>
                </extensions>
              } else {
                <x>{ solution._1 }</x>
              }
            }
          </job>

        resultElement
      }
    }

    override def error(): Elem = {
      withTracer("Unit", this, "error()") {
        <job ref={ taskNode.attribute("id").get }><error>Crosscheck failed.</error></job>
      }
    }

    override def printDescription(): Elem = { <description></description> }

    override def createAlgorithmInstance(parameter: DLProblem): SuitableAlgorithm = {
      withTracer("SuitableAlgorithm", this, "createAlgorithmInstance(parameter: Elem)") {
        val tracer = getCurrentTracer()
        className match {
          case "PohligHellmanWithBSGS" => new PohligHellmanWithBSGS(parameter)
          case "PohligHellmanWithPollardRho" => new PohligHellmanWithPollardRho(parameter)
        }
      }
    }
  }
  
  class GeneratorSearchTask(taskNode: Node, className: String, extensions: Boolean) extends AlgorithmExec[BigInt,BigInt,Elem] {
    type SuitableAlgorithm = GeneratorSearch

    override def input(): BigInt = {
      withTracer("Elem", this, "input()") {
        val tracer = getCurrentTracer()
        val argsNode = (taskNode \ "args").head
        BigInt((argsNode \ "prime").text)
      }
    }
  
    override def createAlgorithmInstance(parameter: BigInt): SuitableAlgorithm = {
      withTracer("SuitableAlgorithm", this, "createAlgorithmInstance(parameter: Elem)") {
        val tracer = getCurrentTracer()
        new GeneratorSearch(parameter)
      }
    }
    
    override def output(): Elem = {
      withTracer("Elem", this, "output()") {
        <job ref={ taskNode.attribute("id").get }>
          <g>{theResult}</g>
        </job>
      }
    }

    override def error(): Elem = {
      withTracer("Unit", this, "error()") {
        <job ref={ taskNode.attribute("id").get }><error>Crosscheck failed.</error></job>
      }
    }

    override def printDescription(): Elem = { <description></description> }
  }
  
  class PMinusOneTask(taskNode: Node, className: String, extensions: Boolean) extends AlgorithmExec[BigInt, Option[BigInt], Elem] {
    type SuitableAlgorithm = PMinusOneMethod
  
    override def input(): BigInt = {
      withTracer("Elem", this, "input()") {
        val tracer = getCurrentTracer()
        val argsNode = (taskNode \ "args").head
        BigInt((argsNode \ "n").text)
      }
    }
  
    override def createAlgorithmInstance(parameter: BigInt): SuitableAlgorithm = {
      withTracer("SuitableAlgorithm", this, "createAlgorithmInstance(parameter: Elem)") {
        val tracer = getCurrentTracer()
        new PMinusOneMethod(parameter)
      }
    }

    override def output(): Elem = {
      withTracer("Unit", this, "output(result: Option[BigInt])") {
        val tracer = getCurrentTracer()

        val solution = theResult match {
          case Some(x) => (x.toString(), true)
          case None => ("No factor found.", false)
        }

        val resultElement =
          <job ref={ taskNode.attribute("id").get }>
            {
              if (extensions && solution._2) {
                <d>{ solution._1 }</d>
                <extensions>
                  <cofactor>
                    {
                      (suitableAlgorithm.cofactor.get).toString()
                    }
                  </cofactor>
                </extensions>
              } else {
                <x>{ solution._1 }</x>
              }
            }
          </job>

        resultElement
      }
    }

    override def error(): Elem = {
      withTracer("Unit", this, "error()") {
        <job ref={ taskNode.attribute("id").get }><error>Crosscheck failed.</error></job>
      }
    }

    override def printDescription(): Elem = { <description></description> }
  }

  object BatchExecutor extends Tracing {
    
    lazy val batchElement = scala.xml.XML.loadFile(new File("." + File.separator + "batch" + File.separator + "pohlighellman-batch.xml"))
    lazy val taskMap = produceTaskMap(batchElement)
  
	  private def produceTaskMap(batchElement: Elem): Map[Int,Node] = {
	    withTracer("Map[Int,Node]", this, "produceTaskMap(batchElement: Elem)") {
		    val taskSeq = batchElement\"task"
		    taskSeq.map(taskNode => {
		      val id = (taskNode\"@id").text.toInt
		      (id, taskNode)
		    }).toMap
	    }
	  }
	  
	  private def task(id: String, node: Node, extensionFlag: Boolean): AlgorithmExec[_,_,Elem] = {
	    id match {
	      case "PohligHellmanWithBSGS" => new PohligHellmanTask(node, id, extensionFlag)
	      case "PohligHellmanWithPollardRho" => new PohligHellmanTask(node, id, extensionFlag)
	      case "GeneratorSearch" => new GeneratorSearchTask(node, id, extensionFlag)
	      case "(p-1)-Method" => new PMinusOneTask(node, id, extensionFlag)
	      case _ => throw new UnsupportedOperationException("Unknown task '" + id + "'")
	    }
	  }
  
	  private def run(batchElement: Elem, taskMap: Map[Int,Node]): Seq[Tuple2[String,NodeSeq]] = {
	    withTracer("Seq[Tuple2[String,NodeSeq]]", this, "run(batchElement: Elem, taskMap: Map[Int,Node])") {
	      val tracer = getCurrentTracer
	      val algorithmSeq = batchElement\"algorithm"
	      val batchResult = algorithmSeq.map(algorithmNode => {
	        tracer.out().printfIndentln("algorithm[class=%s]", (algorithmNode\"@class").text)
	        val jobSeq = algorithmNode\"job"
	        val jobSeqRes = jobSeq.map(jobNode => {
	          try {
				      val refId = (jobNode\"@ref").text.toInt
              val extensionsFlag = (jobNode \ "@extensions").text == "true"
				      tracer.out().printfIndentln("job[refId = %d, extensionsFlag = %b]", refId: Integer, extensionsFlag: java.lang.Boolean)
				      assert(taskMap.get(refId).isDefined, "Referenced task[" + refId + "] doesn't exist.")
				      val exec: AlgorithmExec[_,_,Elem] = task((algorithmNode\"@class").text, taskMap.get(refId).get, extensionsFlag)
				      exec.process
            }
			      catch {
			        case ex: UnsupportedOperationException => {
			          tracer.logException(LogLevel.ERROR, ex, this.getClass(), "run(batchElement: Elem, taskMap: Map[Int,Node])")
			          <job ref={jobNode.attribute("ref")}><error>{ex.getMessage()}</error></job>
			        }
			        case ex: IllegalArgumentException => {
			          tracer.logException(LogLevel.ERROR, ex, this.getClass(), "run(batchElement: Elem, taskMap: Map[Int,Node])")
			          <job ref={jobNode.attribute("ref")}><error>{ex.getMessage()}</error></job>
			        }
			        case er: java.lang.Exception => {
			          tracer.logMessage(LogLevel.FATAL, "An unexpected error has occurred.", this.getClass(), "run(batchElement: Elem, taskMap: Map[Int,Node])")
			          <job ref={jobNode.attribute("ref")}><error>{"An unexpected error has occurred."}</error></job>
			        }
			        case er: AssertionError => {
			          tracer.logMessage(LogLevel.FATAL, er.getMessage(), this.getClass(), "run(batchElement: Elem, taskMap: Map[Int,Node])")
			          <job ref={jobNode.attribute("ref")}><error>{er.getMessage()}</error></job>
			        }
			      }
	        })
	        ((algorithmNode\"@class").text, jobSeqRes)
	      })
	      batchResult
	    }
	  }

    private def produceResultDoc(batchElement: Elem, resultSeq: Seq[Tuple2[String, NodeSeq]]): Elem = {

      def deepCopy(node: Node): Node = {
        node match {
          case elem: Elem => {
            elem.copy(scope = TopScope, child = elem.child.map(c => deepCopy(c)))
          }
          case _ => node
        }
      }

      withTracer("Node", this, "produceResultDoc(batchElement: Elem, resultSeq: Seq[Tuple2[String,NodeSeq]])") {
        val tracer = getCurrentTracer()
        <batch xmlns="http://www.christofreichardt.de/scala/dl-algorithms">
          <timestamp>{ new java.util.Date().toString() }</timestamp>
          {
            (batchElement \ "task").map(task => {
              <task id={ task.attribute("id").get }>
                {
                  deepCopy((task \ "args")(0))
                }
              </task>
            }) ++
              resultSeq.map(result => {
                <algorithm class={ result._1 }>
                  { result._2 }
                </algorithm>
              })
          }
        </batch>
      }
    }

    private def format(node: Node): Node = {
      
	    def removeEmptyTextNodes(node: Node): Node = {
	      node match {
	        case elem: Elem => {
	          val subNodeSeq =
		          elem.child.filter(node => {
		            node match {
		              case text: Text => {
	                  !text.data.trim.isEmpty
		              }
		              case _ => true
		            }
		          })
	          Elem(elem.prefix, elem.label, elem.attributes, elem.scope, elem.minimizeEmpty, subNodeSeq.map(c => removeEmptyTextNodes(c)):_*)
	        }
	        case _ => node
	      }
	    }
	    
	    def addIndentation(node: Node, depth: Int): Node = {
	    
		    def isLeaf(elem: Elem): Boolean = {
		      if (elem.child.size == 1) {
		        elem.child(0) match {
		          case c: Atom[Any] => true
		          case _ => false
		        }
		      }
		      else
		      	false
		    }
	    
        node match {
          case elem: Elem => {
            if (!isLeaf(elem)) {
	            val newSubNodes = (for (subNode <- elem.child) yield List(Text("\n" + "  "*depth), subNode)).flatten ++ List(Text("\n" + "  "*(depth - 1)))
	            Elem(elem.prefix, elem.label, elem.attributes, elem.scope, elem.minimizeEmpty, newSubNodes.map(subNode => addIndentation(subNode, depth + 1)):_*)
            }
            else
              elem
          }
          case _ => node
        }
	    }
	    
      withTracer("Node", this, "format(node: Node)") {
        addIndentation(removeEmptyTextNodes(node), 1)
      }
    }

    def main(args: Array[String]): Unit = {
      val startTime = System.currentTimeMillis()
      TracerFactory.getInstance().readConfiguration(new File("." + File.separator + "config" + File.separator + "tracerfactory-config.xml"))
      val tracer = getCurrentTracer
      tracer.open()
      tracer.initCurrentTracingContext()
      try {
        withTracer("Unit", this, "main(args: Array[String])") {
          val propertyNames = System.getProperties().stringPropertyNames.toArray(new Array[String](0)).sorted
          propertyNames.foreach(propertyName => getCurrentTracer().out().printfIndentln("%s = %s", propertyName, System.getProperty(propertyName)))
          
//          val batchElement = scala.xml.XML.loadFile(new File("." + File.separator + "batch" + File.separator + "pohlighellman-batch.xml"))
//          val taskMap = produceTaskMap(batchElement)
          val batchResult = run(batchElement, taskMap)
          val resultDoc = format(produceResultDoc(batchElement, batchResult))
          scala.xml.XML.save("." + File.separator + "batch" + File.separator + "pohlighellman-batch-result.xml", resultDoc, "UTF-8", true)
        }
      }
      catch {
        case er: java.lang.Throwable => {
          tracer.logException(LogLevel.ERROR, er, this.getClass(), "main(args: Array[String])")
        }
      }
      finally {
        tracer.close()
      }
      val timeToCompletion = (System.currentTimeMillis() - startTime)/1000f
      printf("%n%d Tasks processed in %.3f seconds.%n", taskMap.size, timeToCompletion)
    }

    override def getCurrentTracer(): AbstractTracer = {
      try {
        TracerFactory.getInstance().getCurrentPoolTracer()
      } 
      catch {
        case ex: TracerFactory.Exception => TracerFactory.getInstance().getDefaultTracer
      }
    }
  }
}
