/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package de.christofreichardt.httpmonitor;

import com.sun.net.httpserver.HttpContext;
import com.sun.net.httpserver.HttpServer;
import de.christofreichardt.diagnosis.AbstractTracer;
import de.christofreichardt.diagnosis.LogLevel;
import de.christofreichardt.diagnosis.Traceable;
import de.christofreichardt.diagnosis.TracerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

/**
 *
 * @author Christof Reichardt
 */
public class HttpUnit implements Traceable {
  static public final String CHARSET = "UTF-8";
  static public final int CLIENT_PORTNO = 7070;
  static public final String CONTEXT_PATH = "/r/programming/";
  static public final String SERVER_HOST = "www.reddit.com";
  static public final int SERVER_PORTNO = 80;
  static public final int THREAD_POOL_SIZE = 3;
  
  private ExecutorService executorService;
  
  @Before
  public void init() {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "init()");
    
    try {
      this.executorService = Executors.newFixedThreadPool(THREAD_POOL_SIZE, new ThreadFactory() {
        private final AtomicInteger threadCounter = new AtomicInteger();

        @Override
        public Thread newThread(Runnable runnable) {
          return new Thread(runnable, "Worker-" + this.threadCounter.getAndIncrement());
        }
      });
    }
    finally {
      tracer.wayout();
    }
  }
  
  @After
  public void exit() {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "exit()");
    
    try {
      final long TIMEOUT = 5;
      this.executorService.shutdown();
      try {
        if (!this.executorService.awaitTermination(TIMEOUT, TimeUnit.SECONDS)) {
          this.executorService.shutdownNow();
          if (!this.executorService.awaitTermination(TIMEOUT, TimeUnit.SECONDS))
            tracer.logMessage(LogLevel.WARNING, "Thread pool didn't terminate.", getClass(), "exit()");
        }
      }
      catch (InterruptedException ex) {
        tracer.logException(LogLevel.WARNING, ex, getClass(), "exit()");
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  @Test
  public void simpleGET() throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "simpleGET()");
    
    try {
      URL url = new URL("http://www.reddit.com:80/r/programming/");
      HttpURLConnection urlConnection = (HttpURLConnection) url.openConnection();
      urlConnection.setRequestProperty("Accept-Charset", CHARSET);
      urlConnection.connect();
      int responseCode = urlConnection.getResponseCode();
      String statusline = String.format("HTTP/1.0 %d %s", responseCode, urlConnection.getResponseMessage());
      
      tracer.out().printfIndentln("statusline = %s", statusline);
      Assert.assertTrue(statusline, responseCode == HttpURLConnection.HTTP_OK);
      
      try (InputStream inputStream = urlConnection.getInputStream()) {
        InputStreamReader inputStreamReader = new InputStreamReader(inputStream, Charset.forName(CHARSET));
        LineNumberReader lineNumberReader = new LineNumberReader(inputStreamReader);
        do {
          String line = lineNumberReader.readLine();
          if (line == null)
            break;
          
          tracer.out().printfIndentln("line = %s", line);
        } while(true);
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  @Test
  public void monitorGET() throws IOException, InterruptedException, ExecutionException, TimeoutException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "monitorGET()");

    try {
      HttpServer httpServer = HttpServer.create(new InetSocketAddress(CLIENT_PORTNO), 0);
      
      try {
        httpServer.setExecutor(this.executorService);
        HttpContext httpContext = httpServer.createContext(CONTEXT_PATH);
        httpContext.setHandler(new HttpRelais(SERVER_HOST, SERVER_PORTNO));
        httpServer.start();
        
        Future<Integer> future = this.executorService.submit(new HttpGetTask());
        Integer responseCode = future.get(5, TimeUnit.SECONDS);
        
        Assert.assertTrue("Http error.", responseCode == HttpURLConnection.HTTP_OK);
      }
      finally {
        httpServer.stop(10);
      }
    }
    finally {
      tracer.wayout();
    }
  }

  @Override
  public AbstractTracer getCurrentTracer() {
    return TracerFactory.getInstance().getCurrentPoolTracer();
  }
}
