/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package de.christofreichardt.httpmonitor;

import de.christofreichardt.diagnosis.AbstractTracer;
import de.christofreichardt.diagnosis.LogLevel;
import de.christofreichardt.diagnosis.Traceable;
import de.christofreichardt.diagnosis.TracerFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Date;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSOutput;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.SAXException;

/**
 *
 * @author Christof Reichardt
 */
public class SOAPTracer implements Traceable {
  private final DOMImplementationRegistry domImplRegistry;
  private final LSOutput lsOutput;
  private final LSSerializer serializer;

  public SOAPTracer() {
    this.domImplRegistry = createDOMImplRegistry();
    DOMImplementationLS domImplLS = (DOMImplementationLS) this.domImplRegistry.getDOMImplementation("LS");
    if (domImplLS == null)
      throw new RuntimeException("No Load and Save DOM implementation found.");
    this.lsOutput = domImplLS.createLSOutput();
    this.serializer = domImplLS.createLSSerializer();
    this.serializer.getDomConfig().setParameter("format-pretty-print", Boolean.TRUE);
  }

  private DOMImplementationRegistry createDOMImplRegistry() {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("DOMImplementationRegistry", this, "createDOMImplRegistry()");
    
    try {
      try {
        return DOMImplementationRegistry.newInstance();
      }
      catch (ClassNotFoundException | InstantiationException | IllegalAccessException | ClassCastException | DOMException ex) {
        throw new RuntimeException("Cannot create DOMImplRegistry.");
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  public Document parse(byte[] bytes) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("Document", this, "parse(byte[] bytes)");
    
    try {
      ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(bytes);
      
      DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
      documentBuilderFactory.setValidating(false);
      documentBuilderFactory.setNamespaceAware(true);
      try {
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        return documentBuilder.parse(byteArrayInputStream);
      }
      catch (ParserConfigurationException | SAXException ex) {
        tracer.logException(LogLevel.ERROR, ex, getClass(), "read");
        throw new IOException("Cannot parse given bytes.", ex);
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  public void trace(Document document) {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "trace(Document document)");
    
    try {
      this.lsOutput.setByteStream(tracer.out());
      Date now = new Date();
      tracer.out().printf("--- SOAP BEGIN -----------------------------------------------------------------------------------%n");
      serializer.write(document, lsOutput);
      tracer.out().printf("--- SOAP END -------------------------------------------------------------------------------------%n");
    }
    finally {
      tracer.wayout();
    }
  }

  @Override
  public AbstractTracer getCurrentTracer() {
    return TracerFactory.getInstance().getCurrentPoolTracer();
  }
}
