/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package de.christofreichardt.httpmonitor;

import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import de.christofreichardt.diagnosis.AbstractTracer;
import de.christofreichardt.diagnosis.LogLevel;
import de.christofreichardt.diagnosis.Traceable;
import de.christofreichardt.diagnosis.TracerFactory;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.w3c.dom.Document;

/**
 *
 * @author Christof Reichardt
 */
public class HttpRelais implements HttpHandler, Traceable {
  final private String serverHost;
  final private int serverPortNo;
  final private SOAPTracer soapTracer;

  public HttpRelais(String serverHost, int serverPortNo) {
    this.serverHost = serverHost;
    this.serverPortNo = serverPortNo;
    this.soapTracer = new SOAPTracer();
  }
  
  @Override
  public void handle(HttpExchange httpExchange) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.initCurrentTracingContext();
    tracer.entry("void", this, "handle(HttpExchange httpExchange)");
    
    try {
      final RequestParameter requestParameter = new RequestParameter(httpExchange);
      requestParameter.trace();
      
      try {
        URL url = constructUrl(requestParameter);
        HttpURLConnection urlConnection = (HttpURLConnection) url.openConnection();
        urlConnection.setRequestMethod(requestParameter.getRequestMethod());
        forwardRequestHeaders(requestParameter.getRequestHeaders(), urlConnection);

        if (requestParameter.getRequestMethod().equalsIgnoreCase("POST")) {
          urlConnection.setDoOutput(true);
          forwardRequestBody(httpExchange, urlConnection);

          int responseCode = urlConnection.getResponseCode();
          tracer.out().printfIndentln("responseCode = %d", responseCode);

          forwardResponseHeader(urlConnection, httpExchange, responseCode);
          forwardResponseBody(urlConnection, httpExchange, responseCode);
        }
        else if (requestParameter.getRequestMethod().equalsIgnoreCase("GET")) {
          int responseCode = urlConnection.getResponseCode();
          tracer.out().printfIndentln("responseCode = %d", responseCode);

          forwardResponseHeader(urlConnection, httpExchange, responseCode);
          forwardResponseBody(urlConnection, httpExchange, responseCode);
        }
      }
      catch (IOException ex) {
        tracer.logException(LogLevel.ERROR, ex, getClass(), "handle(HttpExchange httpExchange)");
      }
      catch (Throwable t) {
        tracer.logException(LogLevel.ERROR, t, getClass(), "handle(HttpExchange httpExchange)");
      }
      finally {
        httpExchange.close();
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  private URL constructUrl(RequestParameter requestParameter) throws MalformedURLException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("String", this, "constructUrl(RequestParameter requestParameter)");
    
    try {
      String spec;
      if (requestParameter.getRawPath() != null) {
        spec = "http://" + this.serverHost + ":" + this.serverPortNo + requestParameter.getRawPath();
      }
      else {
        spec = "http://" + this.serverHost + ":" + this.serverPortNo;
      }
      if (requestParameter.getRawQuery() != null)
        spec = spec + "?" + requestParameter.getRawQuery();
      
      tracer.out().printfIndentln("spec = %s", spec);
      
      return new URL(spec);
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void forwardRequestHeaders(Headers requestHeaders, HttpURLConnection urlConnection) {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "forwardRequestHeaders(Headers requestHeaders, HttpURLConnection urlConnection)");
    
    try {
      for (Map.Entry<String, List<String>> entry : requestHeaders.entrySet()) {
        if (!"Host".equals(entry.getKey())) {
          tracer.out().printIndent(entry.getKey() + ": ");
          StringBuilder stringBuilder = new StringBuilder();
          List<String> values = entry.getValue();
          Iterator<String> iter = values.iterator();
          while (iter.hasNext()) {
            String value = iter.next();
            tracer.out().print(value);
            stringBuilder.append(value);
            if (iter.hasNext()) {
              tracer.out().print(", ");
              stringBuilder.append(", ");
            }
          }
          tracer.out().println();
          urlConnection.setRequestProperty(entry.getKey(), stringBuilder.toString());
        }
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void forwardRequestBody(HttpExchange httpExchange, HttpURLConnection urlConnection) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "forwardRequestBody(HttpExchange httpExchange, HttpURLConnection urlConnection)");
    
    try {
      final int BUFFER_SIZE = 256;
      byte[] buffer = new byte[BUFFER_SIZE];
      
      try (InputStream inputStream = httpExchange.getRequestBody();
          ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
        do {
          int read = inputStream.read(buffer);
          tracer.out().printfIndentln("read = %d", read);
          tracer.out().flush();
          if (read == -1) {
            break;
          }
          byteArrayOutputStream.write(buffer, 0, read);
        } while (true);
        
        byte[] bytes = byteArrayOutputStream.toByteArray();
        Document document = this.soapTracer.parse(bytes);
        this.soapTracer.trace(document);
        
        try (OutputStream outputStream = urlConnection.getOutputStream()) {
          outputStream.write(bytes);
        }
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void forwardResponseHeader(HttpURLConnection urlConnection, HttpExchange httpExchange, int responseCode) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "forwardResponseHeader(HttpURLConnection urlConnection, HttpExchange httpExchange, int responseCode)");
    
    try {
      Map<String, List<String>> headerFields = urlConnection.getHeaderFields();
      Headers responseHeaders = httpExchange.getResponseHeaders();
      for (Map.Entry<String, List<String>> entry : headerFields.entrySet()) {
        if (entry.getKey() != null) {
          tracer.out().printIndent(entry.getKey() + ": ");
          responseHeaders.put(entry.getKey(), entry.getValue());
          List<String> values = entry.getValue();
          Iterator<String> iter = values.iterator();
          while (iter.hasNext()) {
            String value = iter.next();
            tracer.out().print(value);
            if (iter.hasNext()) {
              tracer.out().print(", ");
            }
          }
          tracer.out().println();
        }
      }
      httpExchange.sendResponseHeaders(responseCode, 0);
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void forwardResponseBody(HttpURLConnection urlConnection, HttpExchange httpExchange, int responseCode) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "forwardResponseBody(HttpURLConnection urlConnection, HttpExchange httpExchange)");
    
    try {
//      tracer.out().printfIndentln("urlConnection.getDoOutput() = %b", urlConnection.getDoOutput());
//      tracer.out().flush();
      
      final int BUFFER_SIZE = 256;
      byte[] buffer = new byte[BUFFER_SIZE];
      
      try (InputStream inputStream = getInputStreamForResponse(urlConnection, responseCode);
          ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
        do {
          int read = inputStream.read(buffer);
          tracer.out().printfIndentln("read = %d", read);
          tracer.out().flush();
          if (read == -1) {
            break;
          }
          byteArrayOutputStream.write(buffer, 0, read);
        } while (true);
        
        byte[] bytes = byteArrayOutputStream.toByteArray();
        Document document = this.soapTracer.parse(bytes);
        this.soapTracer.trace(document);
        
        try (OutputStream outputStream = httpExchange.getResponseBody()) {
          outputStream.write(bytes);
        }
      }
      
    }
    finally {
      tracer.wayout();
    }
  }
  
  private InputStream getInputStreamForResponse(HttpURLConnection urlConnection, int responseCode) throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "forwardResponseBody(HttpURLConnection urlConnection, HttpExchange httpExchange)");
    
    try {
      InputStream inputStream;
      if (responseCode != 200)
        inputStream = urlConnection.getErrorStream();
      else
        inputStream = urlConnection.getInputStream();
      
      return inputStream;
    }
    finally {
      tracer.wayout();
    }
  }

  @Override
  public AbstractTracer getCurrentTracer() {
    return TracerFactory.getInstance().getCurrentPoolTracer();
  }

}
