/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package de.christofreichardt.httpmonitor;

import com.sun.net.httpserver.HttpContext;
import com.sun.net.httpserver.HttpServer;
import de.christofreichardt.diagnosis.AbstractTracer;
import de.christofreichardt.diagnosis.LogLevel;
import de.christofreichardt.diagnosis.Traceable;
import de.christofreichardt.diagnosis.TracerFactory;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 *
 * @author student
 */
public class App implements Traceable {
  final private HttpServer httpServer;
  final private ExecutorService executorService;
  final private String contextPath;
  final private HttpRelais httpRelais;

  public App() throws IOException {
    Properties properties = init();
    
    if (!properties.containsKey("de.christofreichardt.httpmonitor.serverHost"))
      throw new RuntimeException("Missing 'de.christofreichardt.httpmonitor.serverHost' property.");
    String serverHost = properties.getProperty("de.christofreichardt.httpmonitor.serverHost");
    int serverPortNo = Integer.parseInt(properties.getProperty("de.christofreichardt.httpmonitor.serverPortNo", "8080"));
    this.httpRelais = new HttpRelais(serverHost, serverPortNo);
    
    if (!properties.containsKey("de.christofreichardt.httpmonitor.clientPortNo"))
      throw new RuntimeException("Missing 'de.christofreichardt.httpmonitor.clientPortNo' property.");
    int clientPortNo = Integer.parseInt(properties.getProperty("de.christofreichardt.httpmonitor.clientPortNo"));
    this.httpServer = HttpServer.create(new InetSocketAddress(clientPortNo), 0);
    
    int nThreads = Integer.parseInt(properties.getProperty("de.christofreichardt.httpmonitor.nThreads", "3"));
    this.executorService = Executors.newFixedThreadPool(nThreads, new ThreadFactory() {
      private final AtomicInteger threadCounter = new AtomicInteger();
      @Override
      public Thread newThread(Runnable runnable) {
        return new Thread(runnable, "Worker-" + this.threadCounter.getAndIncrement());
      }
    });
    
    if (!properties.containsKey("de.christofreichardt.httpmonitor.contextPath"))
      throw new RuntimeException("Missing 'de.christofreichardt.httpmonitor.contextPath' property.");
    this.contextPath = properties.getProperty("de.christofreichardt.httpmonitor.contextPath");
  }
  
  private Properties init() throws IOException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "init()");
    
    try {
      Properties properties = new Properties();
      File propertiesFile = new File("." + File.separator + "httpmonitor.properties");
      try (FileInputStream fileInputStream = new FileInputStream(propertiesFile)) {
        properties.load(fileInputStream);
      }
      
      return properties;
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void exit() {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "exit()");
    
    try {
      final long TIMEOUT = 5;
      this.executorService.shutdown();
      try {
        if (!this.executorService.awaitTermination(TIMEOUT, TimeUnit.SECONDS)) {
          this.executorService.shutdownNow();
          if (!this.executorService.awaitTermination(TIMEOUT, TimeUnit.SECONDS))
            tracer.logMessage(LogLevel.WARNING, "Thread pool didn't terminate.", getClass(), "exit()");
        }
      }
      catch (InterruptedException ex) {
        tracer.logException(LogLevel.WARNING, ex, getClass(), "exit()");
      }
    }
    finally {
      tracer.wayout();
    }
  }
  
  private void monitor() throws IOException, InterruptedException {
    AbstractTracer tracer = getCurrentTracer();
    tracer.entry("void", this, "monitor()");
    
    try {
      this.httpServer.setExecutor(this.executorService);
      HttpContext httpContext = this.httpServer.createContext(this.contextPath);
      httpContext.setHandler(this.httpRelais);
      this.httpServer.start();
      
      tracer.out().printfIndentln("Server has been started.");
      tracer.out().flush();
      
      File stopFile = new File("." + File.separator + "stop.txt");
      stopFile.createNewFile();
      final long lastModified = stopFile.lastModified();
      long INTERVAL = 2500;
      do {
        Thread.sleep(INTERVAL);
        System.out.printf("Monitoring ...%n");
        if (stopFile.lastModified() > lastModified)
          break;
      } while (true);
      
      int DELAY = 5;
      this.httpServer.stop(DELAY);
    }
    finally {
      tracer.wayout();
    }
  }

  @Override
  public AbstractTracer getCurrentTracer() {
    return TracerFactory.getInstance().getCurrentPoolTracer();
  }

  /**
   * @param args the command line arguments
   * @throws de.christofreichardt.diagnosis.TracerFactory.Exception
   */
  public static void main(String[] args) throws TracerFactory.Exception {
    TracerFactory.getInstance().reset();
    InputStream resourceAsStream = App.class.getClassLoader().getResourceAsStream("de/christofreichardt/httpmonitor/trace-config.xml");
    if (resourceAsStream != null) {
      TracerFactory.getInstance().readConfiguration(resourceAsStream);
    }
    TracerFactory.getInstance().openPoolTracer();
    
    try {
      AbstractTracer tracer = TracerFactory.getInstance().getCurrentPoolTracer();
      tracer.initCurrentTracingContext();
      tracer.entry("void", App.class, "main(String[] args)");
      try {
        List<String> propertyNames = new ArrayList<>(System.getProperties().stringPropertyNames());
        Collections.sort(propertyNames);
        for (String propertyName : propertyNames) {
          tracer.out().printfIndentln("%s = %s", propertyName, System.getProperties().getProperty(propertyName));
        }
        
        try {
          App app = new App();
          try {
            app.monitor();
          }
          finally {
            app.exit();
          }
          
        }
        catch (Exception ex) {
          tracer.logException(LogLevel.ERROR, ex, App.class, "main(String[] args)");
        }
      }
      finally {
        tracer.wayout();
      }
    }
    finally {
      TracerFactory.getInstance().closePoolTracer();
    }
  }
  
}
